<?php
/*======================================================================*\
|| #################################################################### ||
|| # ---------------------------------------------------------------- # ||
|| # Copyright ©2013 Fillip Hannisdal AKA Revan/NeoRevan/Belazor 	  # ||
|| # All Rights Reserved. 											  # ||
|| # This file may not be redistributed in whole or significant part. # ||
|| # ---------------------------------------------------------------- # ||
|| # You are not allowed to use this on your server unless the files  # ||
|| # you downloaded were done so with permission.					  # ||
|| # ---------------------------------------------------------------- # ||
|| #################################################################### ||
\*======================================================================*/

// #############################################################################
// DBSEO "Custom URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread';
	public static $structure = 'showthread.php?t=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		if (DBSEO::checkHighlight(true))
		{
			// We need to redirect
			DBSEO::safeRedirect($_SERVER['REQUEST_URI'], array('highlight'));
		}

		// Retrieve highlight params
		DBSEO::checkHighlight(false);

		if (!$urlInfo['thread_id'])
		{
			// We're missing a thread ID
			
			if (!isset($urlInfo['forum_id']) AND isset($urlInfo['forum_path']))
			{
				// Grab our forum cache
				$forumcache = DBSEO::$db->fetchForumCache();

				foreach ($forumcache as $forumid => $forum)
				{
					if ($forum['seopath'] == $urlInfo['forum_path'])
					{
						// Discovered the right forum
						$urlInfo['forum_id'] = $forumid;
						break;
					}
				}
			}

			if (!isset($urlInfo['forum_id']) AND isset($urlInfo['forum_title']))
			{
				// Reverse forum title
				$urlInfo['forum_id'] = DBSEO::reverseForumTitle($urlInfo);
			}

			// Reverse lookup thread ID
			$urlInfo['thread_id'] = DBSEO::reverseObject('thread', $urlInfo['thread_title'], $urlInfo['forum_id']);
		}

		return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['thread_id'], $urlInfo['thread_page']);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		if (!count(DBSEO::$cache['rawurls']))
		{
			// Ensure we got this kickstarted
			DBSEO::initUrlCache();
		}

		// Prepare the regexp format
		$format 		= explode('_', (is_null($format) ? self::$format : $format), 2);
		$rawFormat 		= DBSEO::$cache['rawurls'][strtolower($format[0])][$format[1]];

		// Init this
		$replace = array();

		$data['postid'] = intval($data['postid'] ? $data['postid'] : isset($data['p']) ? $data['p'] : 0);
		$data['threadid'] = intval($data['threadid'] ? $data['threadid'] : $data['t']);

		if ($data['postid'])
		{
			// We had a paged blog
			$replace['%post_id%'] = $data['postid'];

			if (!$data['threadid'])
			{
				// We need to extract thread info from post info
				$postInfo = DBSEO::getPostInfo($data['postid']);
				$data['threadid'] = $postInfo['threadid'];
			}			
		}

		if ($data['threadid'])
		{
			// Get our thread info
			$threadInfo = self::getInfo($data['threadid']);
		}

		if (!$threadInfo['threadid'])
		{
			// Forum didn't exist
			return '';
		}

		if ($appendA OR !$threadInfo['seotitle'] OR preg_match('#[-_\s](?:post|print)?\d+$#', $threadInfo['seotitle']))
		{
			// Only do this in certain cases
			$threadInfo['seotitle'] .= ($threadInfo['seotitle'] ? DBSEO::$config['dbtech_dbseo_rewrite_separator'] : '') . 'a';
		}

		// Handle userid and username
		$replace['%thread_id%'] 	= $threadInfo['threadid'];
		$replace['%thread_title%'] 	= $threadInfo['seotitle'];


		$data['forumid'] = intval($threadInfo['forumid']);
		if ($data['forumid'])
		{
			// Grab our forum cache
			$forumcache = DBSEO::$db->fetchForumCache();

			// Grab our forum info
			$forumInfo = DBSEO::$db->cache['forumcache'][$data['forumid']];
		}
		
		if (!$forumInfo['forumid'])
		{
			// Forum didn't exist
			return '';
		}

		// Handle userid and username
		$replace['%forum_id%'] 	= $forumInfo['forumid'];
		$replace['%forum_title%'] = DBSEO::rewriteForumUrl($forumInfo);
		$replace['%forum_path%'] = $forumInfo['seopath'];

		if ($data['page'])
		{
			// We had a paged blog
			$replace['%thread_page%'] = $data['page'];
		}

		// Handle the replacements
		$newUrl = str_replace(array_keys($replace), $replace, $rawFormat);
		
		//if (strpos($newUrl, '%') !== false)
		//{
			// We should not return true if any single URL remains
			//return '';
		//}

		// Ensure we don't double down on this
		$newUrl = str_replace(
			DBSEO::$config['dbtech_dbseo_rewrite_separator'] . DBSEO::$config['dbtech_dbseo_rewrite_separator'], 
			DBSEO::$config['dbtech_dbseo_rewrite_separator'], 
			$newUrl
		);

		// Return the new URL
		return $newUrl;
	}

	/**
	 * Gets any extra information needed
	 *
	 * @param mixed $threadIds
	 * @param boolean $force
	 * 
	 * @return array
	 */
	public static function getInfo($threadIds, $force = false)
	{
		$threadInfo = array();

		if (!is_array($threadIds))
		{
			// Ensure this is an array
			$threadIds = array($threadIds);
		}

		if (!$force)
		{
			foreach ($threadIds as $key => $id)
			{
				if (($info = DBSEO::$datastore->fetch('threadinfo.' . $id)) === false)
				{
					// We don't have this cached
					continue;
				}

				// We had this cached, cache it internally too
				DBSEO::$cache['thread'][$id] = $info;
			}
		}

		$queryList = array();
		foreach ($threadIds as $key => $threadId)
		{
			if (!isset(DBSEO::$cache['thread'][$threadId]) OR $force)
			{
				// Ensure this is done
				$queryList[$key] = intval($threadId);
			}
			else
			{
				if (isset(DBSEO::$cache['thread'][$threadId]['seotitle']))
				{
					// We had a SEO title
					continue;
				}

				/*DBTECH_PRO_START*/
				if (DBSEO::$config['dbtech_dbseo_rewrite_thread_smarttitle'])
				{
					if (!isset(DBSEO::$cache['thread'][$threadId]['firstpostid']))
					{
						// Ensure this is done
						$queryList[$key] = intval($threadId);
						continue;
					}

					// Grab post info
					$postInfo = DBSEO::getPostInfo(DBSEO::$cache['thread'][$threadId]['firstpostid']);

					// Content filter
					DBSEO::$cache['thread'][$threadId]['seotitle'] = DBSEO::contentFilter($postInfo['pagetext']);
				}
				/*DBTECH_PRO_END*/

				if (!DBSEO::$cache['thread'][$threadId]['seotitle'])
				{
					// Normal filter
					DBSEO::$cache['thread'][$threadId]['seotitle'] = DBSEO::filterText(
						DBSEO::$cache['thread'][$threadId]['threadtitle'] ? 
						DBSEO::$cache['thread'][$threadId]['threadtitle'] : 
						DBSEO::$cache['thread'][$threadId]['title']
					);
				}
			}
		}

		if (count($queryList))
		{
			$info = DBSEO::$db->generalQuery('
				SELECT *
				FROM $thread
				WHERE threadid IN (' . implode(',', $queryList) . ')
			', false);
			foreach ($info as $arr)
			{
				// Init this
				$arr['seotitle'] = '';

				/*DBTECH_PRO_START*/
				if (DBSEO::$config['dbtech_dbseo_rewrite_thread_smarttitle'])
				{
					// Grab post info
					$postInfo = DBSEO::getPostInfo($arr['firstpostid']);

					// Content filter
					$arr['seotitle'] = DBSEO::contentFilter($postInfo['pagetext']);
				}
				/*DBTECH_PRO_END*/

				if (!$arr['seotitle'])
				{
					// Normal filter
					$arr['seotitle'] = DBSEO::filterText(
						$arr['threadtitle'] ? 
						$arr['threadtitle'] : 
						$arr['title']
					);
				}

				// Build the cache
				DBSEO::$datastore->build('threadinfo.' . $arr['threadid'], $arr);

				// Cache this info
				DBSEO::$cache['thread'][$arr['threadid']] = $arr;
			}
		}

		if (count($threadIds) == 1)
		{
			// We have only one, return only one
			$threadInfo = DBSEO::$cache['thread'][$threadIds[0]];
		}
		else
		{
			foreach ($threadIds as $key => $threadId)
			{
				// Create this array
				$threadInfo[$threadId] = DBSEO::$cache['thread'][$threadId];
			}
		}

		return $threadInfo;
	}
}


// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_Page extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_Page';
	public static $structure = 'showthread.php?t=%d&page=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}
}


// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_LastPost extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_LastPost';
	public static $structure = 'showthread.php?t=%d&goto=lastpost';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}
}


// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_NewPost extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_NewPost';
	public static $structure = 'showthread.php?t=%d&goto=newpost';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}
}

// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_GoToPost extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_GoToPost';
	public static $structure = 'showthread.php?p=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['post_id'], $urlInfo['page']);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}
}


// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_GoToPost_Page extends DBSEO_Rewrite_Thread_GoToPost
{
	public static $format = 'Thread_Thread_GoToPost_Page';
	public static $structure = 'showthread.php?p=%d&page=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}
}

// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_Previous extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_Previous';
	public static $structure = 'showthread.php?t=%d&goto=nextoldest';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		if (!$thread = parent::getInfo($urlInfo['thread_id']))
		{
			// Couldn't load thread info
			return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['thread_id']);
		}

		if (!$previousThread = self::getInfo($thread))
		{
			// Couldn't load previous thread info
			return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['thread_id']);
		}

		// Git oot
		DBSEO::safeRedirect(self::createUrl($previousThread, 'Thread_Thread'));
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}

	/**
	 * Gets any extra information needed
	 *
	 * @param mixed $thread
	 * @param boolean $force
	 * 
	 * @return array
	 */
	public static function getInfo($thread, $force = false)
	{
		$threadInfo = array();

		if (($info = DBSEO::$datastore->fetch('prevthread.' . intval($thread['forumid']) . '.' . intval($thread['lastpost']))) === false OR $force)
		{
			$info = DBSEO::$db->generalQuery('
				SELECT *
				FROM $thread
				WHERE forumid = ' . intval($thread['forumid']) . '
					AND lastpost < ' . intval($thread['lastpost']) . '
					AND visible = 1
					AND open <> 10
				ORDER BY lastpost DESC
				LIMIT 1
			', false);

			// Build the cache
			DBSEO::$datastore->build('prevthread.' . intval($thread['forumid']) . '.' . intval($thread['lastpost']), $info);
		}

		foreach ($info as $arr)
		{
			if (!isset($arr['seotitle']))
			{
				// Set this
				DBSEO::$cache['thread'][$arr['threadid']]['seotitle'] = $arr['seotitle'] = DBSEO::filterText($arr['title']);
			}

			// Cache this info
			DBSEO::$cache['thread'][$arr['threadid']] = $arr;
		}

		return $arr;
	}
}


// #############################################################################
// DBSEO "Thread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_Thread_Next extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_Thread_Next';
	public static $structure = 'showthread.php?t=%d&goto=nextnewest';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		if (!$thread = parent::getInfo($urlInfo['thread_id']))
		{
			// Couldn't load thread info
			return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['thread_id']);
		}

		if (!$nextThread = self::getInfo($thread))
		{
			// Couldn't load next thread info
			return sprintf((is_null($structure) ? self::$structure : $structure), $urlInfo['thread_id']);
		}

		// Git oot
		DBSEO::safeRedirect(self::createUrl($nextThread, 'Thread_Thread'));
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format);
	}

	/**
	 * Gets any extra information needed
	 *
	 * @param mixed $thread
	 * @param boolean $force
	 * 
	 * @return array
	 */
	public static function getInfo($thread, $force = false)
	{
		$threadInfo = array();

		if (($info = DBSEO::$datastore->fetch('nextthread.' . intval($thread['forumid']) . '.' . intval($thread['lastpost']))) === false OR $force)
		{
			$info = DBSEO::$db->generalQuery('
				SELECT *
				FROM $thread
				WHERE forumid = ' . intval($thread['forumid']) . '
					AND lastpost > ' . intval($thread['lastpost']) . '
					AND visible = 1
					AND open <> 10
				ORDER BY lastpost ASC
				LIMIT 1
			', false);

			// Build the cache
			DBSEO::$datastore->build('nextthread.' . intval($thread['forumid']) . '.' . intval($thread['lastpost']), $info);
		}

		foreach ($info as $arr)
		{
			if (!isset($arr['seotitle']))
			{
				// Set this
				DBSEO::$cache['thread'][$arr['threadid']]['seotitle'] = $arr['seotitle'] = DBSEO::filterText($arr['title']);
			}

			// Cache this info
			DBSEO::$cache['thread'][$arr['threadid']] = $arr;
		}

		return $arr;
	}
}


// #############################################################################
// DBSEO "PrintThread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_PrintThread extends DBSEO_Rewrite_Thread
{
	public static $format = 'Thread_PrintThread';
	public static $structure = 'printthread.php?t=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format, true);
	}
}

// #############################################################################
// DBSEO "PrintThread URL" class

/**
* Lets you construct & lookup custom URLs
*/
class DBSEO_Rewrite_PrintThread_Page extends DBSEO_Rewrite_PrintThread
{
	public static $format = 'Thread_PrintThread_Page';
	public static $structure = 'printthread.php?t=%d&page=%d';

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function resolveUrl($urlInfo = array(), $structure = NULL)
	{
		// Determine if we have a structure
		$structure = is_null($structure) ? self::$structure : $structure;

		// Now create the URL
		return parent::resolveUrl($urlInfo, $structure);
	}

	/**
	 * Creates a SEO'd URL based on the URL fed
	 *
	 * @param string $url
	 * @param array $data
	 * 
	 * @return string
	 */
	public static function createUrl($data = array(), $format = NULL, $appendA = false)
	{
		// Determine if we have a format
		$format = is_null($format) ? self::$format : $format;

		// Now create the URL
		return parent::createUrl($data, $format, true);
	}
}